//////////////////////////////////////////////
// main.cpp
//
//////////////////////////////////////////////

/// Includes ---------------------------------

// nkAstraeus
#include <NilkinsAstraeus/Log/LogManager.h>

#include <NilkinsAstraeus/Scripts/ScriptsEngine.h>

#include <NilkinsAstraeus/System.h>

// nkLog
#include <NilkinsLog/Loggers/ConsoleLogger.h>

// nkGraphics
#include <NilkinsGraphics/Log/LogManager.h>

#include <NilkinsGraphics/RenderContexts/RenderContextDescriptor.h>
#include <NilkinsGraphics/RenderContexts/RenderContextManager.h>

#include <NilkinsGraphics/System.h>

// nkScripts
#include <NilkinsScripts/Environments/Environment.h>

#include <NilkinsScripts/Interpreters/Interpreter.h>

#include <NilkinsScripts/Scripts/Script.h>
#include <NilkinsScripts/Scripts/ScriptManager.h>

// Standards
#include <memory>

/// Function ---------------------------------

int main ()
{
	// Preparing logging
	std::unique_ptr<nkLog::Logger> logger = std::make_unique<nkLog::ConsoleLogger>() ;
	nkAstraeus::LogManager::getInstance()->setReceiver(logger.get()) ;

	// Launch the engine through dedicated call
	if (!nkAstraeus::System::getInstance()->initialize())
		return -1 ;

	// nkGraphics has been initialized, we can address it directly
	// Create a context to get a window
	nkGraphics::RenderContext* context = nkGraphics::RenderContextManager::getInstance()->createRenderContext(nkGraphics::RenderContextDescriptor(800, 600, false, true)) ;
	nkGraphics::System::getInstance()->run(context) ;

	// And access the environment provided by nkAstraeus, offering a wrapping of the nkEngine
	nkScripts::Script* script = nkScripts::ScriptManager::getInstance()->createOrRetrieve("script") ;
	script->setTargetInterpreter(nkScripts::INTERPRETER::LUA) ;

	script->setSources
	(
		R"eos(
			-- Address the C++ API from Lua
			-- Create two textures
			local tex = nkGraphics.TextureManager.get("environment") ;
			local irr = nkGraphics.TextureManager.createOrRetrieve("irradiance") ;

			-- Compute the irradiance map and load it
			nkGraphics.TextureUtils.computeIrradianceSH(tex, irr) ;
			irr:load() ;

			-- And save it, for instance
			nkGraphics.TextureSaver.saveToHdr(irr, "Textures/environmentIrradiance.hdr") ;
		)eos"
	) ;

	script->load() ;

	nkAstraeus::System::getInstance()->getScriptsEngine()->getEnvironment()->execute(*script) ;

	// Don't forget to properly clean the memory of the engine by shutting it down
	nkAstraeus::System::getInstance()->shutdown() ;
}